/** @file   DataWarehouse.h
 * @brief   Declaration of DataWarehouse class.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/02/21 20:45:42 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_DATAWAREHOUSE_H
#define H_ENG2D_DATAWAREHOUSE_H

#include "eng2d_dll.h"
#include <map>


namespace eng2d {

// Forward declarations.
class BaseContainer;


/** @class  DataWarehouse
 * @brief   A central storage for animations, bitmaps and other objects.
 * @author  Tomi Lamminsaari
 *
 */
class DLLIMPORT DataWarehouse
{
public:

    ///
    /// Constants, datatypes and static methods
    /// =======================================

    typedef std::map<int, BaseContainer*>::const_iterator Iterator;

    ///
    /// Constructors, destructor and operators
    /// ======================================

    /** Default constructor.
     */
    DataWarehouse();

    /** Destructor
     */
    virtual ~DataWarehouse();

private:
  
    /** Copy constructor
     * @param   aAnother          Reference to another DataWarehouse
     */
    DataWarehouse( const DataWarehouse& aAnother );

    /** An assignment operator
     * @param   aAnother          Reference to another DataWarehouse
     * @return  Reference to this object.
     */
    DataWarehouse& operator= ( const DataWarehouse& aAnother );

public:
  

    ///
    /// Inherited methods
    /// =================


    ///
    /// New public methods
    /// ==================

    /** Adds new item to this warehouse. Addition fails if there already exists
     * an item with the same key as the new item.
     * @param   aUid            The key that identifies the data being added.
     * @param   aItem           Pointer to item being added. Ownership is
     *                          transferred to warehouse. If addition fails,
     *                          this item-object is deleted.
     * @return  KErrNone if successfull or KErrAlreadyExists if key is in use.
     */
    int AddItem( int aUid, BaseContainer* aItem );

    /** Deletes the contents of this warehouse.
     */
    void Reset();


    ///
    /// Getter methods
    /// ==============

    /** Returns the item that has been associated to given key.
     * @param   aUid            Id of the item we try to return.
     * @return  Pointer to item that is associated to given key. Nullpointer
     *          if key was not found. Ownership is not transferred to the
     *          caller.
     */
    BaseContainer* Find( int aUid ) const;
    
    /** Returns an iterator that points to the first item of this warehouse.
     * @return  Iterator to the very first item
     */
    Iterator Begin() const;
    
    /** Returns an iterator that points to the end of the warehouse.
     * @return  End iterator.
     */
    Iterator End() const;

protected:
    ///
    /// Protected interface
    /// ===================


protected:
    ///
    /// Data members
    /// ============

    /** Datastructure where the data is kept. */
    std::map<int, BaseContainer*> iItems;
};

};

#endif
